
/* Copyright (c) Miguel Angel Sepulveda, 1998. */

/* This program is freely distributable without licensing fees 
   and is provided without guarantee or warrantee expressed or 
   implied. This program is -not- in the public domain. */

#include <iostream.h> 
#include <math.h> 
#include "Gd_definitions.h" 
#include "Gd_noise.h" 
#include "Gd_monomer.h" 
#include "Gd_polymer.h" 
 
 
/********************************************************************\  
*         GdPolymer          M E M B E R S  
\********************************************************************/  
long unsigned  GdPolymer::seed = 81534857;  
 
GdPolymer::GdPolymer(){  
  numberMonomers = 0;  
  monomer = 0;  
};  
 
 
GdPolymer::GdPolymer(int PolymerLength){  
  // Allocate polymer array  
  numberMonomers = PolymerLength;  
  monomer = new GdMonomer[PolymerLength];  
	  
  // Link monomers:  
  // bonds are implemented as "arrows", if it goes from a to b  
  //  only a accounts for the bond not b  
  for (int i = 0; i < numberMonomers -1; i++)  
    monomer[i].makeBond(monomer[i+1]);  
};  
 
 
 
/* Return number of monomers in polymer */ 
int GdPolymer::getSize() const{ 
  return numberMonomers; 
}; 
 
 
/* Return i-th monomer of the polymer */ 
GdMonomer& GdPolymer::operator[](int i){ 
  return monomer[i]; 
}; 
 
 
GdPolymer::GdPolymer(GdPolymer &orig){ 
  numberMonomers = orig.numberMonomers; 
 
  if(monomer != 0) delete[] monomer; 
  monomer = new GdMonomer[numberMonomers]; 
   
  for (int j = 0; j < numberMonomers; j++)  
    monomer[j] = orig.monomer[j]; 
 
  for (int i = 0; i < numberMonomers -1; i++)  
    monomer[i].makeBond(monomer[i+1]);  
   
}; 
 
 
GdPolymer::~GdPolymer(){  
  delete[]  monomer;  
};  
 
 
double GdPolymer::bondLength() const {  
  double bonds = 0.0;  
	  
  for (int i = 0; i < numberMonomers-1; i++)  
    bonds += monomer[i].distance(monomer[i+1]);  
	  
  if(numberMonomers != 1) bonds /= (numberMonomers-1);   
  return bonds;  
};  
 
 
double GdPolymer::getVolume() const {  
  static double constant = 4.0 /3.0 * pi;  
  double bonds = 0.0;  
	  
  for (int i = 0; i < numberMonomers-1; i++)  
    bonds += monomer[i].distance(monomer[i+1]);  
	  
  double radius = 0.5 * bonds;  
  double volume = constant * pow(radius, 3.0);  
	  
  return volume;  
};  
 
void GdPolymer::resetVelocity(){  
  for (int i = 0; i < numberMonomers; i++){  
    monomer[i].resetVelocity();  
  };  
};  
 
 
void GdPolymer::resetAcceleration(){  
  for (int i = 0; i < numberMonomers; i++){  
    monomer[i].resetAcceleration();  
  };  
};  
 
 
void GdPolymer::linear(double BondLength) {  
  double TotalLength = (numberMonomers-1) * BondLength;  
  double x;  
  double y = 0.0;  
  double z = 0.0;  
  x = -TotalLength/2.0;  
	  
  for (int i = 0; i < numberMonomers; i++){  
    monomer[i].setPosition(x, y, z);  
    x += BondLength;  
  };	  
};  
 
 
void GdPolymer::center(double &xc, double &yc, double &zc) const {  
  xc = yc = zc = 0.0;  
  for (int i = 0; i < numberMonomers; i++){  
    double x, y, z;  
    monomer[i].getPosition(x, y, z);  
    xc += x;  
    yc += y;  
    zc += z;  
  };  
	  
  xc /= numberMonomers;  
  yc /= numberMonomers;  
  zc /= numberMonomers;  
};  
 
void GdPolymer::translate(double xt, double yt, double zt){  
	  
  for(int i = 0; i < numberMonomers; i++)  
    monomer[i].addPosition(xt, yt, zt);  
	  
};  
 
 
void GdPolymer::random(unsigned long &seed, double length){  
  // Choose a random orientation   
  double phi = 2.0 * pi * ran1(&seed);  
  double theta = pi * (-1.0 + 2.0 * ran1(&seed));  
  double xc = cos(theta) * cos(phi);  
  double yc = cos(theta) * sin(phi);  
  double zc = sin(theta);  
	  
  // Compute monomer coordinates  
  double L = length * (numberMonomers -1);  
  double offset = -0.5 * L;  
  for (int i = 0; i < numberMonomers; i++) {  
    double x, y, z;  
    x = offset * xc;  
    y = offset * yc;  
    z = offset * zc;  
    monomer[i].setPosition(x, y, z);  
    offset += length;  
  };  
};  
 
 
 
void GdPolymer::moveTo(double xt, double yt, double zt) {  
  int i;  
	  
  // Get center of mass  
  double xc, yc, zc;  
  xc = yc = zc = 0.0;  
  for (i = 0; i < numberMonomers; i++){  
    double x, y, z;  
    monomer[i].getPosition(x, y, z);  
    xc += x;  
    yc += y;  
    zc += z;  
  };  
	  
  xc /= numberMonomers;  
  yc /= numberMonomers;  
  zc /= numberMonomers;  
	  
	  
  // Next find translation vector 
  xt -= xc;  
  yt -= yc;  
  zt -= zc;  
	  
  // Transform coordinates  
  for(i = 0; i < numberMonomers; i++)  
    monomer[i].addPosition(xt, yt, zt);  
	  
};  
 
 
GdPolymer& GdPolymer::operator=(GdPolymer &p) {  
  numberMonomers = p.numberMonomers;  
	  
  // Allocate memory for polymer  
  delete [] monomer;  
  monomer = new GdMonomer[numberMonomers];  
	  
  // Copy coordinates, velocities and accelerations  
  for (int j = 0; j < numberMonomers; j++)  
    monomer[j] = p.monomer[j]; 
 
  // Build polymer structure  
  for(int i = 0; i < numberMonomers-1; i++)  
    monomer[i].makeBond(monomer[i+1]);  
	 	 	  
  return *this;  
};  
 
 
ostream& operator<<(ostream &cs, GdPolymer &data){  
  for (int i = 0; i < data.numberMonomers; i++){  
    cs << "monomer[" << i << "] ";   
    cs << data.monomer[i] << endl;  
  };  
	  
  return cs;  
};  
 

